(* ::Subsubsection:: *)
(*HoughtonCassegrain package*)

(* ::Text:: *)
(*This impements the Houghton-Cassegrain telescope optical system*)

(* ::Text:: *)
(*Chapter*)

(* ::Text:: *)
(*This file should be in the following path*)

(* ::Text:: *)
(*FileNameJoin[{$UserBaseDirectory, "Applications/GeometricOptics/HoughtonCassegrain"}]*)

BeginPackage["GeometricOptics`Packages`HoughtonCassegrain`", {"GeometricOptics`"}]

Options[HoughtonCassegrain] = {OutputType->"Report", OutputLevel->"Full", CameraType->"SD"};
(* 	OutputType can be 
					"Report", generates a new report (notebook) for each call to the function
					"Print", prints results in the current position of the evaluating notebook
					"Basic", gives the list of results
					"Values" or any other directive, gives the list of values (no names of variables)
	OutputLevel can be
					"Minimal", gives only output values, those calculated by the function
					"Full", gives the list of input parameters and output values
	CameraType can be
					"SD" Houghton's telescope with symmetric doublet
					"DDR" Houghton's telescope with doublet with different radii - no control of astigmatism
*)

HoughtonCassegrain::usage="HoughtonCassegrain calculates parameters for a Houghton-Cassegrain telescope optical system.";
HoughtonCassegrainInputPanel::usage = "HoughtonCassegrainInputPanel gives the GUI panel to work with HoughtonCassegrain";

$HoughtonCassegrainInputVariables = {"f1", "ft", "spes", "em", "diam", "\[Delta]", "Ni", "\[Theta]"};

Begin["`Private`"]

HoughtonCassegrain[{f1_, ft_, spes_, em_, diam_, delta_, Ni_, theta_}, opts___]:= HoughtonCassegrain[f1, ft, spes, em, diam, delta, Ni, theta];

HoughtonCassegrain[f1_, ft_, spes_, em_, diam_, delta_, Ni_, theta_, OptionsPattern[]] := 
Quiet@Module[{\[Alpha], M, \[Beta], \[Gamma], thick, A, B, AS, r11, r12, rad, a1, a2, D1, D2, r3, rd, r1, r2, r3d, r4, \[CapitalDelta], n, inputs, outputs, inPanel, outPanel, title},
 		If[ArgumentsQ["HoughtonCassegrain", {$HoughtonCassegrainInputVariables, {f1, ft, spes, em, diam, delta, Ni, theta}}],
			AppendTo[$ExamplesStack, 
					 <|"PackageID" -> 8, "PackageName" -> "HoughtonCassegrain", "Arguments" -> <|"f1" -> f1, "ft" -> ft, "spes" -> spes, "em" -> em, "diam" -> diam, "\[Delta]" -> delta, "Ni" -> Ni, "\[Theta]" -> theta|>|>];
			$ExamplesStack = DeleteExampleDuplicates[$ExamplesStack];
		
			\[Alpha] = em/f1;
			M = ft/ f1;
			\[Beta] = (M - \[Alpha])/(M + 1);
			\[Gamma] = (2 M (1 + \[Alpha]))/(1 - M^2);
			thick = Join[spes, {delta f1, -\[Beta] f1}];
		
			Switch[OptionValue[CameraType],
					"SD",
					A = -(1/(32 f1^3)) + ((\[Alpha] + 1) (-1 + M^2))/(32 f1^3 M^3);
					B = (\[Alpha] (1 - M^2) + M (1 + M^2))/(8 f1^2 M^3) - ((1 - M^2 + M^3 + \[Alpha] - M^2 \[Alpha]) delta)/(8 f1^2 M^3);
					AS = (M^4 - 2 M \[Alpha] - \[Alpha]^2 - 2 M^3 (2 + \[Alpha]) + M^2 (-1 + \[Alpha]^2))/(8 f1 M^3 (1 + \[Alpha])) + ((M + M^3 + \[Alpha] - M^2 \[Alpha]) delta)/(4 f1 M^3) - ((1 + M^3 + \[Alpha] - M^2 (1 + \[Alpha])) delta^2)/(8 f1 M^3);
					r11 = (2 A B (-1 + Ni) (1 + Ni))/(-2 A^2 - 2 A^2 Ni + B^3 Ni - B^3 Ni^2);
					r12 = (2 A B (-1 + Ni^2))/(2 A^2 + 2 A^2 Ni + B^3 Ni - B^3 Ni^2);
					rad = {r11, r12, -r11, -r12};
					TotalAberrations[{r11, r12, -r11, -r12, -2f1, \[Gamma] f1}, thick, {{1, Ni, 1, Ni, 1, -1, 1}}, {0, 0, 0, 0, 0, 0, 0}, diam/2, 0, 0, -Infinity, x, theta, {\[Lambda]}],
					
					"DDR",
					A = -(1/(32 f1^3)) + ((\[Alpha] + 1) (-1 + M^2))/(32 f1^3 M^3);
					B = (\[Alpha] (1 - M^2) + M (1 + M^2))/(8 f1^2 M^3) - ((1 - M^2 + M^3 + \[Alpha] - M^2 \[Alpha]) delta)/(8 f1^2 M^3);
					AS = (M^4 - 2 M \[Alpha] - \[Alpha]^2 - 2 M^3 (2 + \[Alpha]) + M^2 (-1 + \[Alpha]^2))/(8 f1 M^3 (1 + \[Alpha])) + ((M + M^3 + \[Alpha] - M^2 \[Alpha]) delta)/(4 f1 M^3) - ((1 + M^3 + \[Alpha] - M^2 (1 + \[Alpha])) delta^2)/(8 f1 M^3);
					a1 = (4 - 3 Ni - 3 Ni^2 + 2 Ni^3)/(8 Ni);
					a2 = (-1 + Ni + Ni^2 - Ni^3)/(4 Ni);
					D1 = 1/8 - 1/(4 Ni) + Ni/8 - ((-1 + Ni^2) r3 f1 A)/(2 Ni B);
					D2 = (2 Ni)/(Ni^2 - 1) B*r3^2 f1^2;
					\[CapitalDelta] = (a1 + a2 + D1)^2 - 4 (a1 + a2) (D1 + a1 D2);
					r3 = Last@Sort@ReplaceAll[r3, Solve[Numerator[Simplify[\[CapitalDelta]]] == 0, r3]];
					r3 = 0.99 r3; 
					(* Second solution *)
					r1 = -f1 r3 (2 a1 (a1 + a2))/(a1 (a2 - 2 D1) + a2 (a2 - D1 + Sqrt[\[CapitalDelta]])); 
					r2 = f1 r3 (2 (a1 + a2))/(a1 + a2 + D1 + Sqrt[\[CapitalDelta]]);
					r3d = f1 r3;
					r4 = (r1 r2 r3d)/ (r1 (r2 - r3d) + r2 r3d); 
					rad = {r1, r2, r3d, r4};
					TotalAberrations[{r1, r2, r3d, r4, -2 f1, \[Gamma] f1}, thick, {{1, Ni, 1, Ni, 1, -1, 1}}, {0, 0, 0, 0, 0, 0}, diam/2, 0, 0, -Infinity, x, theta, {\[Lambda]}]
				 ];
			(* Drawing *)
			fun = MapThread[If[#1 > 0, #1 + #2 - Sqrt[#1^2 - x^2], #1 + #2 + Sqrt[#1^2 - x^2]] &, {rad, Prepend[Accumulate[spes], 0]}];
			gr = Plot[Evaluate@fun, {x, -diam/2, diam/2}, PlotStyle -> {Red, Red, Blue, Blue}, AspectRatio -> 1/5, PlotRange -> All];
			  	  
			outputs = {{"Primary radius", -2 f1},
					   {"Secondary radius", \[Gamma] f1},
					   {"Distance between the secondary and primary", -\[Beta] f1},
					   {"Obstruction coefficient", 1-\[Beta]},
					   {"The final radii of Houghton's doublet are",""},
					   {"r1", rad[[1]]},
					   {"r2", rad[[2]]},
					   {"r3", rad[[3]]},
					   {"r4", rad[[4]]},
					   {"Height of the image", GOimageHeight[[1, GOn]]},
					   {"Focal length", GOfocalLength[[1]]},
					   {"Total spherical aberration", GOaberration[[1]]},
					   {"Total sagittal coma", GOcoma[[1]]},
					   {"Total astigmatism", GOastigmatism[[1]]},
					   {"Plots", gr}};
			
			(* defines the two panels, input parameters and output values *)
			inPanel = Grid[{{"Focal length of the primary mirror" , "f1", f1},
							{"Total focal length", "ft", ft},
							{"List of thickness of the doublet", "spes", spes},
							{"Back distance", "em", em},
							{"Diameter of the corrector", "diam", diam},
							{"Distance of doublet from the mirror", "\[Delta]", delta},
							{"Refractive index of corrector", "Ni", Ni},
							{"Field angle  in degree", "\[Theta]", theta},
							{"Camera Type","", OptionValue[CameraType]}},
							Alignment -> {{Left, Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"InputParameterBottom"}];
  
			outPanel = Grid[outputs, 
							Alignment -> {{Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"OutputValueBottom"}];

			cameratype = Switch[OptionValue[CameraType], 
								"SD",
								"Symmetrix doublet - no control of astigmatism",
								"DDR",
								"Camera with doublet with different radii - no control of astigmatism"];
			title = BoxData[ToBoxes[Column[{$GeometricOpticsPackagesList[SelectFirst[#PackageName == "HoughtonCassegrain" &], "Description"], Style[cameratype, "Text"]}]]];
			
			(* generates the type of output required *)
			Switch[OptionValue[OutputType],
					"Report",
					GenerateDocument[TemplateApply[$ReportTemplate, 
										Join[<|	"title" -> title,																	
												"date" -> DateString[], 
												"function" -> "HoughtonCassegrain", 
												"outputlevel" -> OptionValue[OutputLevel],
												"inPanel" -> inPanel, 
												"outPanel" -> outPanel |>]]];,

					"Print",
					CellPrint[TextCell[TemplateApply[$PrintTemplate, 
											Join[<|	"title" -> title, 
													"date" -> DateString[], 
													"function" -> "HoughtonCassegrain", 
													"outputlevel" -> OptionValue[OutputLevel],
													"inPanel" -> inPanel, 
													"outPanel" -> outPanel |>]], "Text"]];,
					"Basic",
					CellPrint[TextCell[
					TemplateApply[$BasicTemplate, 
									Join[<| "outputlevel" -> OptionValue[OutputLevel],
											"inputs" -> {{"f1", f1},
														 {"ft", ft},
														 {"spes", spes},
														 {"em", em},
														 {"diam", diam},
														 {"\[Delta]", delta},
														 {"Ni", Ni},
														 {"\[Theta]", theta}},
											"outputs" -> outputs |>]], "Output"]],
					_,
					CellPrint[TextCell[DeleteCases[outputs[[All, 2]], Alternatives["", Style[___]]], "Output"]]],

							
			(* Arguments are not correct *)
			MessageDialog["HoughtonCassegrain not executed, the number or the type of the arguments may be incorrect.", WindowTitle->"Warning: example not generated"];]];

HoughtonCassegrainInputPanel[]:=
DynamicModule[{outputtype = "Report", package = "HoughtonCassegrain"},
	examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
						Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
    examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], Alignment -> Center, Spacings -> {1, 1}, Dividers -> Center], FrameStyle -> LightGray] &, examplesAll];
	example = "Browse...";
	Panel[Column[{	DynamicWrapper[Style[NameFromPackageName[package] <> " Input Panel", "Subsection"], 
									If[NumericQ[example], {f1, ft, spes, em, diam, delta, Ni, theta} = ReplaceAll[$HoughtonCassegrainInputVariables, examplesAll[[example]]]]],
					Style["Insert values for each argument, then use Evaluate to run HoughtonCassegrain function", "Text"],
					Row[{
					Grid[{{"Focal length of the primary mirror" , "f1", Tooltip[InputField[Dynamic[f1], Alignment -> Center], "Insert the focal length of the primary mirror"]},
						  {"Total focal length", "ft", Tooltip[InputField[Dynamic[ft], Alignment -> Center], "Insert the total focal length"]},
						  {"List of thickness of the doublet", "spes", Tooltip[InputField[Dynamic[spes], Alignment -> Center], "Insert the list of thickness of the doublet"]},
						  {"Back distance", "em", Tooltip[InputField[Dynamic[em], Alignment -> Center], "Insert the back distance"]},
						  {"Diameter of the primary mirror", "diam", Tooltip[InputField[Dynamic[diam], Alignment -> Center], "Input the value of diam"]},
						  {"Distance between the corrector and the mirror", "\[Delta]", Tooltip[InputField[Dynamic[delta], Alignment -> Center], "Input the distance between the corrector and the mirror"]},
						  {"Refractive index of the corrector", "Ni", Tooltip[InputField[Dynamic[Ni], Alignment -> Center], "Input the refractive index of the corrector"]},
						  {"Field angle in degrees", "\[Theta]", Tooltip[InputField[Dynamic[theta], Alignment -> Center], "Insert the value of \[Theta]"]}},
						Spacings -> {1, 0},
						Alignment -> {{Left, Left, Right}, Center}, 
						Dividers -> Center, 
						FrameStyle -> LightGray],
					Column[{"Camera type", PopupMenu[Dynamic[cameratype], {"SD", "DDR"}]}]}],
					RadioButtonBar[Dynamic[cameratype], {"SD" -> "Simmetric doublet", "DDR" -> "Doublet different radii "}],
					Spacer[5],					
					OpenerView[{"Load an example from the archives (current session and saved DB)",
								Row[{Dynamic@PopupMenu[Dynamic[example], Thread[Rule[Range[Length[examples]], examples]], If[examples === {}, "No example saved", "Browse..."], FrameMargins -> 3, Alignment -> Center],
									 Spacer[5],
									 Button["Update list", (examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
																				Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
															examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], 
																						Alignment -> Center, 
																						Spacings -> {1, 1}, 
																						Dividers -> Center], 
																						FrameStyle -> LightGray]&, examplesAll];
															example = "Browse..."), Method -> "Queued"]}]}, 
								Alignment -> Center, 
								Spacings -> 1.5],
					Row[{"Define the type of output to generate", 
						 Spacer[5],
						 RadioButtonBar[Dynamic[outputtype], {"Report" -> Tooltip["Report", "Generates a new notebook reporting a summary of the calculation"], 
															  "Print" -> Tooltip["Print", "Print the table of the calculation done inside the current notebook"], 
															  "Basic" -> Tooltip["Basic", "Generate a list of computed output with label"], 
															  "Values" -> Tooltip["Values", "Return only the list of output values"]}]}],
					Row[{Button["Evaluate", ToExpression[package][Apply[Sequence, {f1, ft, spes, em, diam, delta, Ni, theta}], OutputType -> outputtype, CameraType->cameratype], Method -> "Queued"],
						 Button["Clear all", Map[Clear, Unevaluated[{f1, ft, spes, em, diam, delta, Ni, theta}]]]}]}, 
				Spacings -> 2, 
				Alignment -> Center],
	BaseStyle -> {InputFieldBoxOptions -> {FieldSize -> {15, 1}}}]];
			
  
  End[]
  EndPackage[]